#!/usr/bin/env bash
set -euo pipefail

# ----------------------------------------------------------------------
# Master orchestrator for Volume-4 Fractal-Pivot Hypersurface Pipeline
# ----------------------------------------------------------------------
REPO_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
CONFIG="$REPO_DIR/configs/default.yaml"
RESULTS_DIR="$REPO_DIR/data/results"
KERNEL_DIR="$REPO_DIR/data/kernels"
FLIP_RESULTS_DIR="$RESULTS_DIR/vol4_loop_fluctuation_sim"
FLIP_COUNTS_DIR="$REPO_DIR/data/flip_counts"

die() { echo "ERROR: $*" >&2; exit 1; }

echo "🔧 Building kernels from eigenpairs..."
python "$REPO_DIR/kernel_builder/make_kernel_from_eigs.py" \
  --config "$CONFIG" \
  --eigs "$REPO_DIR/kernel_builder/kernel_eigs.csv" \
  --output-dir "$KERNEL_DIR"
# Cleanup any leftover generic kernels
rm -f "$KERNEL_DIR/kernel.csv" "$KERNEL_DIR/kernel.npy"

echo "🔄 Generating per-L flip counts..."
rm -rf "$FLIP_RESULTS_DIR"
for L in 6 8 10 12 16 20; do
  echo "  • L = $L"
  python "$REPO_DIR/vol4_loop_fluctuation_sim/run.py" \
    --config "$CONFIG" \
    --lattice-size "$L" \
    --output-dir "$FLIP_RESULTS_DIR/L${L}"
done

echo "📂 Slicing flip-counts into data/flip_counts/"
rm -rf "$FLIP_COUNTS_DIR"
mkdir -p "$FLIP_COUNTS_DIR"
for L in 6 8 10 12 16 20; do
  src="$FLIP_RESULTS_DIR/L${L}/flip_counts_L${L}.npy"
  dst="$FLIP_COUNTS_DIR/flip_counts_L${L}.npy"
  [[ -f "$src" ]] || die "missing $src"
  cp "$src" "$dst"
  echo "  copied $src -> $dst"
done

echo "▶️  Running remaining simulations..."
for sim in \
  vol4_discrete_gauge_wilson_loop \
  vol4_loop_interference \
  vol4_string_tension_multi_gauge_low_sweep \
  vol4_wilson_loop_adjoint_volume_sweep \
  vol4_wilson_loop_pipeline_crossover_analysis \
  vol4_wilson_loop_pipeline_lattice_sweep \
  vol4_mass_gap_sim
do
  echo "  • $sim"
  python "$REPO_DIR/$sim/run.py" \
    --config "$CONFIG" \
    --output-dir "$RESULTS_DIR/$sim"
done

echo "🧹 Final cleanup of generic kernels"
rm -f "$KERNEL_DIR/kernel.csv" "$KERNEL_DIR/kernel.npy"

echo "✅ All Volume-4 simulations completed. Results in $RESULTS_DIR"
